#ifndef __HARDWARE_STATE_H
#define __HARDWARE_STATE_H
 
#ifdef __cplusplus 
extern "C" {  
#endif 
  
/* 

	HALs by Mauro Grassi, November 2010.


*/
	
#define IS_PC_HOST						0

#define SIZE_OF_TEMPSTRING				64

#define LONG_ADDRESSES					1
#define REALTIME_PERIOD					183
#define VM_MAX_ARGUMENTS				5

#if(LONG_ADDRESSES)

#define MAX_CACHE_SIZE					0x00FFFFFF
#define MAX_FILE_CACHE_SIZE				0x7FFFFFFF

/* Global Variable Memory Map */

#define GLOBAL_VM_UPPER					0x00001000
#define GLOBAL_VARS_UPPER				0x00002000			/* above this we access the FILE cache */

#else

#define MAX_CACHE_SIZE					0xFFFF
#define MAX_FILE_CACHE_SIZE				0xFFFF

/* Global Variable Memory Map */

#define GLOBAL_VM_UPPER					0x1000
#define GLOBAL_VARS_UPPER				0x2000			/* above this we access the FILE cache */

#endif

#define DEFAULT_BAUD_RATE				115200
#define MAX_NUM_VM						8
#define DEFAULT_VM_EXEC_LIMIT			512
#define MAX_VM_PERIOD					0xFFFFFFFF

#define ROM_CACHE_ID					0x00
#define RAM_CACHE_ID					0x40
#define STACK_CACHE_ID					0x80
#define FILE_CACHE_ID					0xC0

#define VM_ID_MASK						0x3F
#define VM_ID_HOLD						0x80
#define INVALID_ID						0xFF

#define STACK_CACHE_SIZE				64
#define ROM_CACHE_SIZE					512
#define RAM_CACHE_SIZE					128
#define FILE_CACHE_SIZE					128

#define VM_SYSTEM_LOGFILE_CACHE_SIZE	80

#define CACHE_ROM_OFFSET				sizeof(DATA_LOGGER)
#define CACHE_RAM_OFFSET				(CACHE_ROM_OFFSET+ROM_CACHE_SIZE)
#define CACHE_STACK_OFFSET				(CACHE_RAM_OFFSET+RAM_CACHE_SIZE)
#define CACHE_FILE_OFFSET				(CACHE_STACK_OFFSET+STACK_CACHE_SIZE)
#define CACHE_SYSTEM_OFFSET				(CACHE_FILE_OFFSET+FILE_CACHE_SIZE)
#define GENERAL_MEMORY_SIZE				(STACK_CACHE_SIZE+ROM_CACHE_SIZE+RAM_CACHE_SIZE+FILE_CACHE_SIZE+VM_SYSTEM_LOGFILE_CACHE_SIZE+sizeof(DATA_LOGGER))
  
/* VM Print Pipes */

#define NUMDEC_MASK					0x000F	
#define PIPE_MASK					0xFFF0
#define INTERNAL_PIPES_MASK			0x00F0

/* Internal Pipes */

#define NO_PIPE_MODE				0x0000
#define FILENAME_PIPE_MODE			0x0010
#define NMEA_MATCH_STRING_MODE		0x0020
#define NMEA_STRING_MODE			0x0040

/* General Pipes */
#define SYSTEM_LOGFILE_PIPE_MODE	0x1000
#define SERIAL_USB_PIPE_MODE		0x2000
#define LOGFILE_PIPE_MODE			0x4000
#define SERIAL_PIPE_MODE			0x8000	 

#define DEFAULT_PIPE_MODE			(SERIAL_USB_PIPE_MODE | LOGFILE_PIPE_MODE | SERIAL_PIPE_MODE)

/* Type of Implied Argument Modes */

#define NORMAL_TYPE						0
#define TIME_TYPE						1
#define DEFAULT_TYPE_MODE				(NORMAL_TYPE)

/* PF commands, converting types to strings */

#define PF_NOP									0
#define PF_LOCAL_TIME							1
#define PF_LOCAL_TIME_IF_SET					2
#define PF_LOCAL_TIME_FILE_STRING				3
#define PF_LOCAL_DATE_FILE_STRING				4
#define PF_LOCAL_TIME_FILE_STRING_NUMERIC		5
#define PF_LOCAL_DATE_FILE_STRING_NUMERIC		6
#define PF_LOCAL_TIME_OR_DURATION				7
#define PF_LOCAL_DURATION_SINCE_LAST_SYNC		8
#define PF_VM_TIME_FUTURE						9
#define PF_NMEA									10
#define PF_VM_FILENAME							11
#define PF_VM_TIME								17
#define PF_VM_TIME_IF_SET						18
#define PF_VM_TIME_FILE_STRING					19
#define PF_VM_DATE_FILE_STRING					20
#define PF_VM_TIME_FILE_STRING_NUMERIC			21
#define PF_VM_DATE_FILE_STRING_NUMERIC			22
#define PF_VM_TIME_OR_DURATION					23
#define PF_VM_DURATION_SINCE_LAST_SYNC			24
#define PF_GPS_PIPE								25
#define PF_GPS_TEMPSTRING						26

/* The following is the print function for system logging */

#define DEFAULT_LOG_PF							(PF_LOCAL_TIME_OR_DURATION)

/* VM Environment Meta Opcodes */

#define VMENV_NOP				0x00
#define VMENV_START				0x40
#define VMENV_STOP				0x80
#define VMENV_PAUSE				0xC0

/* VM States */
 
#define VM_RUN_MODE				0
#define VM_SLEEP_MODE			1
#define VM_HALT_MODE			2
#define VM_LIMIT_MODE			3

/* VM Environment States */

#define VME_SUPER_TASK_LIMIT	2048

#define VME_START_CYCLE			0
#define VME_GET_NEXT_VM			1
#define VME_STEP_VM				2
#define VME_SUSPEND_VM			3
#define VME_END_CYCLE			4
#define VME_TIMEOUT_CYCLE		5
#define VME_TIMEOUT_WAIT_STATE	6
#define VME_HOLDING_STATE		7
#define VME_HOLDING_ACK_STATE	8
#define VME_WAITING_STATE		9

/* VM environment Modes */

#define VME_NONE				0x00
#define VME_HOLD				0x01
#define VME_DEBUG				0x02
#define VME_VEDEBUG				0x04
#define VME_USE_SLEEP			0x08
#define VME_RESTART				0x10
#define VME_REWIND				0x20

#define VME_DEFAULT_MODE		(VME_USE_SLEEP)

#define VM_OK					0x00
#define CACHE_ERROR				0x01
#define VM_ADDRESS_ERROR		0x02
#define VM_ARG_ERROR			0x03
#define VM_STACK_ERROR			0x04
#define VM_STACK_REF_ERROR		0x05
#define VM_OUT_MEM_ERROR		0x06
#define VM_ERROR				0x07
#define VM_CRC_ERROR			0x08
#define VM_LIMIT				0x09
#define CACHE_ERROR_SEEK_READ	0x10
#define CACHE_ERROR_READ		0x11
#define CACHE_ERROR_SEEK_WRITE	0x20
#define CACHE_ERROR_WRITE		0x21

/* The different Cache Modes */

#define CACHE_MODE_NONE			0x00		/* the basic mode */
#define CACHE_MODE_STATS		0x01		/* Enable Statistics */
#define CACHE_DIRTY				0x02		/* Internally used to determine when a write is needed */
#define CACHE_READ_ONLY			0x04		/* when set, the cache is read only, no writes occur */
#define CACHE_MODE_FILENAME		0x08		/* set this when using a custom filename for the cache file */
#define CACHE_CREATE_ALWAYS		0x10		/* set this so that the cache file is created when the cache is initialized, truncates any other data */
#define CACHE_MODE_LIMIT_UPPER  0x20		/* limits the upper write to the endpointer , use for file log caches */
#define CACHE_APPEND_ALWAYS		0x40

#define FILECACHE_MODE			(CACHE_MODE_NONE | CACHE_MODE_FILENAME |  CACHE_APPEND_ALWAYS | CACHE_MODE_LIMIT_UPPER)
 
#if(IS_PC_HOST)

#define I2C_BUFFER_SIZE						16
#define ONE_WIRE_RESPONSE_BUFFER_SIZE		16
#define BAUD_RATE_FACTOR					10.0

#else

#define I2C_BUFFER_SIZE						16
#define ONE_WIRE_RESPONSE_BUFFER_SIZE		16

#endif

/* Auto Configuration Options */

#define OPEN_SERIAL_PORT			0x01
#define OPEN_I2C_PORT				0x02
#define OPEN_ONE_WIRE_PORT			0x04
			
/* For the PPS Module */

#define NO_PIN						0xFF

/* For the One Wire Bus */

#define ONE_WIRE_PIN_MASK			0x0F
#define ONE_WIRE_USING_IO			0x10
#define ONE_WIRE_OVERDRIVE			0x20
#define NO_ONE_WIRE_MODE			0x80

#define ONE_WIRE_READ_DIR			0x01
#define ONE_WIRE_WRITE_DIR			0x02
#define ONE_WIRE_STRONG_PULLUP		0x04
 
/* Current Use of UART */

#define UART_NO_TX_INV		0x01	/* Invert Tx control */
#define UART_NO_RX_INV		0x02	/* Invert Rx control */
#define UART_OD				0x04	/* Open Collector */
#define UART_RX_INT			0x08	/* Rx Interrupt Enabled */
#define UART_NO_TX			0x10	/* disable Tx */
#define UART_NO_RX			0x20	/* disable Rx */
#define UART_GPS_MODE		0x40	/* enable automatic GPRMC sentence decoding... */
#define NO_UART_MODE		0x80

/* SPI Module Configuration */
#define SPI_BUFFER_SIZE		16

#define SPI_INV_CKE			0x01
#define SPI_INV_CKP			0x02
#define SPI_SPEED_DIV_64	0x04
#define SPI_SPEED_DIV_16	0x08
#define SPI_SPEED_DIV_8		0x10
#define SPI_OPEN_DRAIN		0x20
#define SPI_SAMPLE_AT_END	0x40
#define SPI_NO_MODE			0x80

#define AUTO_MATCH_NUM_BYTES_GPRMC	32
#define RPOR_VALUE_FOR_TX_UART		(6)

#define USE_VM_REALTIME					1
#define VM_ENVIRONMENT_DEBUG			0

#define GLOBAL_ADDRESS_SHIFT			7
#define GLOBAL_ADDRESS_OFFSET			0x7F

#define VM_FILENAME_MAX					31
#define LOG_FILE_NAME_SIZE				24

#define SIZE_OF_DATA_LOGGER_HEADER			(8+sizeof(TIME_T))
// #define SIZE_OF_VM_ENVIRONMENT_HEADER	(5+MAX_NUM_VM+LOG_FILE_NAME_SIZE+sizeof(CACHE_OBJECT))
// #define NUM_WRITE_SETTINGS				(SIZE_OF_DATA_LOGGER_HEADER+SIZE_OF_VM_ENVIRONMENT_HEADER)
#define NUM_WRITE_SETTINGS					(SIZE_OF_DATA_LOGGER_HEADER)

/* Defaults */

#define VM_RECOVERY_TIME			1	/* This is in seconds, a minimum upper bound on how long the system takes to recover from sleep */
#define VM_MINIMUM_SLEEP_PERIOD		5	/* must be greater than the time above */

#define VM_EXEC_PRIORITY_DEFAULT	0xFFFF

#define VM_EXEC_MODE_NO_RESTART		0x01
#define VM_EXEC_MODE_DEFAULT		0x00

/* Abstract Logical Data for UART */

#include "clock.h"

#if(IS_PC_HOST)

#pragma pack(push, 1)

#else

#include "gps.h"

#endif

typedef struct
{
	unsigned char  		mode; 
	unsigned char  		txrxpin;
	unsigned short 		baudRate;
} UART_DESCRIPTOR;

/* Abstract Logical Data for the I2C Bus */

typedef struct
{
	unsigned short busRate;
	unsigned char  I2CBuffer[I2C_BUFFER_SIZE];
} I2C_DESCRIPTOR;

typedef struct
{
	unsigned char  mode;		/* use IO pin or UART ? */
	unsigned char  romCode[8];  /* rom CODE for the device */
	unsigned char  oneWireBuffer[ONE_WIRE_RESPONSE_BUFFER_SIZE];
} ONE_WIRE_DESCRIPTOR;

typedef struct
{
	unsigned char  mode;
	unsigned short ckdidocspin;
	unsigned char  SPIBuffer[SPI_BUFFER_SIZE];
} SPI_DESCRIPTOR;

typedef struct 
{
	/*  
		This object stores information about the state of some of the Hardware
		Resources...
	*/

	UART_DESCRIPTOR 	serialPort;			/* The Current Configuration */
	UART_DESCRIPTOR 	oneWireSerialPort;	/* The Current One Wire Configuration */
	I2C_DESCRIPTOR		i2cPort;			/* The Current I2C Configuration */
	ONE_WIRE_DESCRIPTOR oneWirePort;		/* The One Wire Configuration */
} HARDWARE_DESCRIPTOR;

#if(LONG_ADDRESSES)
typedef unsigned long			ADDRESS_TYPE;
typedef unsigned char			VM_RETURN_TYPE;
#else
typedef unsigned short			ADDRESS_TYPE;
typedef unsigned char			VM_RETURN_TYPE;
#endif
  
typedef struct CCH_TYPE
{
	ADDRESS_TYPE   cacheSize;				/* size in RAM */
	ADDRESS_TYPE   cacheMaxAddress;			/* maximum size on disk */
	ADDRESS_TYPE   bottomAddress;			/* bottom address of RAM segment */
	ADDRESS_TYPE   topAddress;				/* top address of RAM segment */
 	     
#if(IS_PC_HOST)
	unsigned char* 	cacheStart;				/* pointer to RAM start of cache */
#else                  
	unsigned char* 	cacheStart;
	unsigned short 	filler;
#endif
	
	unsigned char   cacheMode;				/* turn off or on the stats */
  	unsigned char   cacheID;				/* unique identifier */
	unsigned long	cacheEndPointer;		/* for appending to the cache easily */	
	union 
	{ 
		unsigned char* cacheFileName;
	 	float 	  	   cachePerformance;	/* performance stats */
	} cacheArgs;
	unsigned short 	cacheHits;				/* performance stats */
	unsigned short 	cacheTotal;				/* performance stats */
} CACHE_OBJECT;
      
typedef struct
{  
	/* 
	
		The virtual machine is organised as follows... 
		
	*/	
	unsigned char  			execID;				/* unique VM Identifier */
	unsigned char  			execState;			/* the current state of the VM */
	unsigned char  			execMode;			/* execution mode */
	unsigned char  			CRC;				/* for consistency! */
	unsigned char  			lastError;			/* last error */
	unsigned char  			IR;					/* instruction register */
	float		   			W;					/* Accumulator */
	TIME_T		   			time;				/* time information for the VM, note that the wday field is used as a pointer too */
	ADDRESS_TYPE   			DSLIMIT;			/* DS LIMIT */
	ADDRESS_TYPE   			DS;					/* Destination String Pointer */
	ADDRESS_TYPE  	 		SS;					/* Source String Pointer */
	unsigned char  			addressModes;		/* Address Modes for the Two Pointers DS and SS */
	ADDRESS_TYPE   			PC;					/* Program Counter */
	ADDRESS_TYPE   			StackSizePtr;		/* Stack Size/Ptr */
	ADDRESS_TYPE   			lastIndirect;		/* last indirect memory accessed, used by the pop indirect instruction */
	unsigned short 			execLimit;			/* limited number of instructions if non zero */
	unsigned short 			execDone;			/* counts the instructions done.. */
	unsigned short			execPriority;		/* a new priority value */
	ADDRESS_TYPE  		 	resetVector;		/* the reset vector */
 	CACHE_OBJECT   			RAMImage;			/* RAM for local variables */
 	CACHE_OBJECT   			ROMImage;			/* code for local program */
 	CACHE_OBJECT   			STACKImage;			/* stack for local program */
	CACHE_OBJECT   			FILEImage;			/* general file cache */
	unsigned short 			pipeMode;							/* print Mode arguments and Pipe Redirection   RRRR RRRR RRRR bbbb */
	unsigned long  			sleepMode;							/* sleep instruction duration storage */
	unsigned char  			fName[VM_FILENAME_MAX];				/* the filename */
	unsigned char  			temp;								/* used internally for short periods */
	unsigned char  			typeMode;							/* type of the implied argument, W or TIME W ? */
	HARDWARE_DESCRIPTOR 	hardwareState;
	GPS_DATA_OBJECT			GPS;								/* GPS data */
	SPI_DESCRIPTOR			SPI;								/* SPI data */
	float					timeSpeed;							/* Used For Time Scaling */
	unsigned long			timeOffset;							/* Used For Time Scaling */
	unsigned char			timeScaling;						/* Used For Time Scaling */
	unsigned char			firstPor;							/* for POR logic */
	unsigned long			tempPipe;							/* used internally for pipe redirection */
} VIRTUAL_MACHINE; 
  
typedef struct
{ 
	unsigned char 	 		vmPtr;								/* current pointer */
	unsigned char   		vmNum;								/* total number of registered VMs */
	unsigned char    		vmState;							/* state of the VM environment FSM */
	unsigned char	 		vmMode;								/* the mode */
	unsigned char    		vmID[MAX_NUM_VM];					/* the queue */
	unsigned char			vmSelected;							/* current selected */
	CACHE_OBJECT	 		vmLogFileCache;						/* for system logging */
	unsigned char	 		vmLogFileName[LOG_FILE_NAME_SIZE];	/* log filename */	
	unsigned long    		vmSleepPeriod[MAX_NUM_VM];			/* the running periods among tasks */
	unsigned long    		vmMinimumPeriod;					/* the cycle minimum period */
	unsigned short   		vmExecLimit;						/* execution limit for multitasking */
	unsigned char			vmRecoveryTime;						/* in seconds */
	unsigned char	 		vmMinimumSleepPeriod;				/* in seconds */	
	VIRTUAL_MACHINE  		cVM;								/* current, active VM */
	float					timeSpeed;
	unsigned long			timeOffset;
	unsigned char			timeScaling;
	HARDWARE_DESCRIPTOR	 	systemState;
} VIRTUAL_MACHINE_ENVIRONMENT;

unsigned char clearUARTDescriptor(UART_DESCRIPTOR* ud);
unsigned char clearI2CDescriptor(I2C_DESCRIPTOR* id);
unsigned char clearOneWireDescriptor(ONE_WIRE_DESCRIPTOR* od);
unsigned char clearHardwareDescriptor(HARDWARE_DESCRIPTOR* hd);
unsigned char compareUARTDescriptor(UART_DESCRIPTOR* system, UART_DESCRIPTOR* newDescriptor, unsigned char mode);
unsigned char compareI2CDescriptor(I2C_DESCRIPTOR* system, I2C_DESCRIPTOR* newDescriptor, unsigned char mode);
unsigned char compareOneWireDescriptor(ONE_WIRE_DESCRIPTOR* systemOW, UART_DESCRIPTOR* systemSerialPort, ONE_WIRE_DESCRIPTOR* newOW, UART_DESCRIPTOR* newSerialPort, unsigned char mode);
unsigned char compareDescriptor(VIRTUAL_MACHINE* vm, HARDWARE_DESCRIPTOR* system, HARDWARE_DESCRIPTOR* newDescriptor, unsigned char mode);
unsigned char autoConfigureState(VIRTUAL_MACHINE* vm, HARDWARE_DESCRIPTOR* system, HARDWARE_DESCRIPTOR* newDescriptor, unsigned char mode);

#if(IS_PC_HOST)

#pragma pack(pop)

#else

#endif

#ifdef __cplusplus
} 
#endif 
 
#endif
